{ TAMAutoRun version 1.2

Did you ever want a component that you can just drop on your form to register your app to run
automatically when Windows starts up?

Look no further, here it is.

There are three different ways to use this tiny component:

A. Drop it on your form, set the AppItemName-property (it must be unique) it can have spaces,
   leave AutoRegister-property true and thats all you have to do

B. Drop it on your form, set the AppItemName-property, set AutoRegister-property to false
   and call, when you want to (un)register you app to start next time windows starts, with the
   following methods:
     if not AMAutoRun.Register then ShowMessage('Not possible to register your app...');
     if not AMAutoRun.UnRegister then ShowMessage('Not possible to unregister your app...');

C. Drop it on your form, set the AppItemName-property, drop a CheckBox-control on your form,
   set the CheckBox-property of this component to direct to the CheckBox and when you check/uncheck
   the checkbox your app will be registered/unregistered. And when your app starts it will check if
   your app is register, if so it will set the CheckBox to checked


Version 1.0 - First release
Version 1.1 - Second release, when you call the (Un)Register-methods and have set the CheckBox-
              property, the CheckBox control will display the current status directly (Checked when
              registered and unchecked when unregistered).
Version 1.2 - Third release, you can now set the 'RunType' property, with Windows there are five
              different types for running an application when booting your system:
              * Run
              * RunOnce
              * RunOnceEx
              * RunServices
              * RunServicesOnce

Use it as you like, when you have a question please mail me: ameeder@dds.nl
}

unit AMAutoRun;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs, Registry, StdCtrls, RegStr;

type
  TRunType = (rtRun, rtRunOnce, rtRunOnceEx, rtRunServices, rtRunServicesOnce);

  TAMAutoRun = class(TComponent)
  private
    FAppItemName: string;
    FAutoRegister: boolean;
    FCheckBox: TCheckBox;
    FRunType: TRunType;

    procedure SetAppItemName(Value: string);
    procedure SetAutoRegister(Value: boolean);
    procedure SetCheckBox(Value: TCheckBox);
    procedure SetRunType(Value: TRunType);
    function GetRegistered: boolean;
  protected
    procedure DoClick(Sender: TObject);
    function RegMe(AutoRun: boolean): boolean;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
    procedure Loaded; override;
  public
    function Register: boolean;
    function Unregister: boolean;
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property Registered: boolean read GetRegistered;
  published
    property AppItemName: string read FAppItemName write SetAppItemName;
    property AutoRegister: boolean read FAutoRegister write SetAutoRegister default true;
    property CheckBox: TCheckBox read FCheckBox write SetCheckBox;
    property RunType: TRunType read FRunType write SetRunType default rtRun;
  end;

procedure Register;

implementation

const
  // Not available in 'RegStr'-unit
  REGSTR_PATH_RUNONCEEX = 'Software\Microsoft\Windows\CurrentVersion\RunOnceEx';

var
  RunKey: string;
  RunTypes : array[TRunType] of string = (REGSTR_PATH_RUN, REGSTR_PATH_RUNONCE, REGSTR_PATH_RUNONCEEX,
                                          REGSTR_PATH_RUNSERVICES, REGSTR_PATH_RUNSERVICESONCE);

procedure TAMAutoRun.DoClick(Sender: TObject);
begin
  case CheckBox.Checked of
    true  : Self.Register;
    false : Self.Unregister;
  end;
end;

procedure TAMAutoRun.SetAppItemName(Value: string);
begin
  if Value <> FAppItemName then FAppItemName := Value;
end;

procedure TAMAutoRun.SetAutoRegister(Value: boolean);
begin
  if (Value <> FAutoRegister) and not Assigned(FCheckBox) then FAutoRegister := Value;
end;

procedure TAMAutoRun.SetCheckBox(Value: TCheckBox);
begin
  if Value <> FCheckBox then
  begin
    FCheckBox := Value;
    if FCheckBox <> nil then FCheckBox.OnClick := DoClick;
    FAutoRegister := False;
  end;
end;

procedure TAMAutoRun.SetRunType(Value: TRunType);
begin
  if Value <> FRunType then
  begin
    FRunType := Value;
    RunKey := RunTypes[Value];
  end;
end;

function TAMAutoRun.Register: boolean;
begin
  Result := RegMe(True);
  if Assigned(FCheckBox) and Result then FCheckBox.Checked := True;
end;

function TAMAutoRun.Unregister: boolean;
begin
  Result := RegMe(False);
  if Assigned(FCheckBox) and Result then FCheckBox.Checked := False;
end;

procedure TAMAutoRun.Notification(AComponent: TComponent; Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (AComponent = FCheckBox) then FCheckBox := nil;
end;

function TAMAutoRun.RegMe(AutoRun: boolean): boolean;
var
  Reg: TRegistry;
begin
  Result := False;
  if AppItemName = '' then Exit;
  Reg := TRegistry.Create;
  try
    Reg.RootKey := HKEY_LOCAL_MACHINE;
    Reg.OpenKey(RunKey, True);
    case AutoRun of
      true  : if not Reg.ValueExists(AppItemName) then
              try
                Reg.WriteString(AppItemName, Application.ExeName);
                Result := True;
              except
                Result := False;
              end;
      false : if Reg.ValueExists(AppItemName) then
              try
                Reg.DeleteValue(AppItemName);
                Result := True;
              except
                Result := False;
              end;
    end;
  finally
    Reg.Free;
  end;
end;

function TAMAutoRun.GetRegistered: boolean;
var
  Reg: TRegistry;
begin
  Result := False;
  if AppItemName <> '' then
  begin
    Reg := TRegistry.Create;
    try
      Reg.RootKey := HKEY_LOCAL_MACHINE;
      Reg.OpenKey(RunKey, True);
      Result := Reg.ValueExists(AppItemName);
    finally
      Reg.Free;
    end;
  end;
end;

procedure TAMAutoRun.Loaded;
var
  Reg: TRegistry;
begin
  inherited Loaded;
  RunKey := RunTypes[RunType];
  if not (csDesigning in ComponentState) and Assigned(FCheckBox) then
  begin
    Reg := TRegistry.Create;
    try
      Reg.RootKey := HKEY_LOCAL_MACHINE;
      Reg.OpenKey(RunKey, True);
      CheckBox.Checked := Reg.ValueExists(AppItemName);
    finally
      Reg.Free;
    end;
  end;
end;

constructor TAMAutoRun.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FAutoRegister := True;
  FRunType := rtRun;
end;

destructor TAMAutoRun.Destroy;
begin
  if AutoRegister and not (csDesigning in ComponentState) then Self.Register;
  inherited Destroy;
end;

procedure Register;
begin
  RegisterComponents('AM Software', [TAMAutoRun]);
end;

end.
